function mrout = ynicmrParse(mr)
% Parse an mr object specification, returning a struct array.
% Useful for flexible argument specification.
%
% mr = mrParse(mr);
%
% The mr argument can be specified as (1) an mr struct; (2) a path to an
% mr file; or (3) a cell of (1) or (2). This function disambiguates the
% specification format and always returns an mr struct array.
%
% ras, 10/2005.
if ~iscell(mr)
    if ischar(mr)
        mrout = mrLoad(mr);
        % YNiC hack.  We know the nifti has had the canonical orientation
        % applied and this routine (ynicmrParse) is only called from
        % mrInit, so we reorient the data appropriately
        if regexp(mr, 'nii\.gz$')
            mrout = fixUpNifti(mrout);
        end
    elseif ~isstruct(mr),
        help(mfilename);
        error('mr is specified in the wrong format');
    end
else
    hwait = waitbar(0, 'Loading mr files ...');

    for i = 1:length(mr)
        if ischar(mr{i})
            mrout{i} = mrLoad(mr{i});
            % YNiC hack.  We know the nifti has had the canonical orientation
            % applied and this routine (ynicmrParse) is only called from
            % mrInit, so we reorient the data appropriately
            if regexp(mr{i}, 'nii\.gz$')
                mrout{i} = fixUpNifti(mrout{i});
            end
        elseif ~isstruct(mr{i}),
            help(mfilename);
            close(hwait);
            error('mr{%i} is specified in the wrong format\n',i);
        end
        tmp(i) = mrout{i};

        waitbar(i/length(mr), hwait);
    end
    mrout = tmp;

    close(hwait);
end

return

function ni = fixUpNifti(ni)

% Let's do something dumb until we get the NIFTI files to store
% the freq_dim, etc headers
% We assume that the slice direction will have the smallest
% number of entries and work it out from that

% Make sure we only work with the first three dims
sizeinfo = size(ni.data);
sizeinfo = sizeinfo(1:3);
[~, mindim] = min(sizeinfo);

% Dimensions are: LR, AP, SI

if mindim == 3
    % Axial layout
    if ndims(ni.data) == 3
        ni.data = permute(ni.data, [2, 1, 3]);
        ni.data = flipdim(ni.data, 1);
    else
        ni.data = permute(ni.data, [2, 1, 3, 4]);
        ni.data = flipdim(ni.data, 1);
    end
    ni.voxelSize = ni.voxelSize([2, 1, 3, 4]);
    ni.dims = ni.dims([2, 1, 3, 4]);
    ni.extent = ni.extent([2, 1, 3, 4]);
elseif mindim == 2
    % Coronal layout
    if ndims(ni.data) == 3
        ni.data = permute(ni.data, [3, 1, 2]);
        ni.data = flipdim(ni.data, 1);
    else
        ni.data = permute(ni.data, [3, 1, 2, 4]);
        ni.data = flipdim(ni.data, 1);
    end

    ni.voxelSize = ni.voxelSize([3, 1, 2, 4]);
    ni.dims = ni.dims([3, 1, 2, 4]);
    ni.extent = ni.extent([3, 1, 2, 4]);
else
    error('I have not bothered implementing saggital support yet');
end

% Some of our old fMRI data has the pixdim (TR) set in ms instead of s
if length(ni.voxelSize) >= 4
    disp('fMRI data has a TR of longer than 1000; assuming ms');
    if ni.voxelSize(4) >= 1000
        ni.voxelSize(4) = ni.voxelSize(4) / 1000.0;
        ni.hdr.pixdim(4) = ni.hdr.pixdim(4) / 1000.0;
    end
end

% TODO: Query - should we update .spaces too?
%       I don't think it's actually used anywhere else
return
