function C = dtiPrePostMult(A, B)
%
% C = dtiPrePostMult(A, B)
%
% Implements C = A*B*A' for dt6 arrays.
%
% Input:
%   B       XxYxZx6xN array of symmetric matrices.
%   A       XxYxZx6xN array of symmetric matrices or
%               XxYxZx3x3xN array of general matrices.
%
% Output:
%   C       XxYxZx6xN array of symmetric matrices.
%

% HISTORY:
%   2004.02.16 ASH (armins@stanford.edu) wrote it.
%   2005.03.03 ASH added multiplication by general matrices.

sz = size(B);
if (ndims(B)>5 | ndims(A)>6),
    error('Wrong input format');
end
if size(B,4)~=6, error('Wrong input format'); end
p = size(A,4);
if (p~=3 & p~=6), error('Wrong input format'); end

if (p==6),
    A = dti6to33(A);
end
A = permute(A, [4:6 1:3]);
B = permute(dti6to33(B), [4:6 1:3]);
C = ndfun('mult', A, ndfun('mult', B, permute(A, [2 1 3:6])));
C = dti33to6(permute(C, [4:6, 1:3]));

return


% The following is a slower but safer implementation
% in the sym*sym*sym case.
C = zeros(size(B));

C(:,:,:,1,:) = ...
    B(:,:,:,1,:).*A(:,:,:,1,:).*A(:,:,:,1,:) + ...
    B(:,:,:,2,:).*A(:,:,:,4,:).*A(:,:,:,4,:) + ...
    B(:,:,:,3,:).*A(:,:,:,5,:).*A(:,:,:,5,:) + ...
    2*B(:,:,:,4,:).*A(:,:,:,1,:).*A(:,:,:,4,:) + ...
    2*B(:,:,:,5,:).*A(:,:,:,1,:).*A(:,:,:,5,:) + ...
    2*B(:,:,:,6,:).*A(:,:,:,4,:).*A(:,:,:,5,:);

C(:,:,:,2,:) = ...
    B(:,:,:,1,:).*A(:,:,:,4,:).*A(:,:,:,4,:) + ...
    B(:,:,:,2,:).*A(:,:,:,2,:).*A(:,:,:,2,:) + ...
    B(:,:,:,3,:).*A(:,:,:,6,:).*A(:,:,:,6,:) + ...
    2*B(:,:,:,4,:).*A(:,:,:,4,:).*A(:,:,:,2,:) + ...
    2*B(:,:,:,5,:).*A(:,:,:,4,:).*A(:,:,:,6,:) + ...
    2*B(:,:,:,6,:).*A(:,:,:,2,:).*A(:,:,:,6,:);

C(:,:,:,3,:) = ...
    B(:,:,:,1,:).*A(:,:,:,5,:).*A(:,:,:,5,:) + ...
    B(:,:,:,2,:).*A(:,:,:,6,:).*A(:,:,:,6,:) + ...
    B(:,:,:,3,:).*A(:,:,:,3,:).*A(:,:,:,3,:) + ...
    2*B(:,:,:,4,:).*A(:,:,:,5,:).*A(:,:,:,6,:) + ...
    2*B(:,:,:,5,:).*A(:,:,:,5,:).*A(:,:,:,3,:) + ...
    2*B(:,:,:,6,:).*A(:,:,:,6,:).*A(:,:,:,3,:);

C(:,:,:,4,:) = ...
    B(:,:,:,1,:).*A(:,:,:,1,:).*A(:,:,:,4,:) + ...
    B(:,:,:,2,:).*A(:,:,:,2,:).*A(:,:,:,4,:) + ...
    B(:,:,:,3,:).*A(:,:,:,5,:).*A(:,:,:,6,:) + ...
    B(:,:,:,4,:).*(A(:,:,:,1,:).*A(:,:,:,2,:) + A(:,:,:,4,:).*A(:,:,:,4,:)) + ...
    B(:,:,:,5,:).*(A(:,:,:,1,:).*A(:,:,:,6,:) + A(:,:,:,4,:).*A(:,:,:,5,:)) + ...
    B(:,:,:,6,:).*(A(:,:,:,2,:).*A(:,:,:,5,:) + A(:,:,:,4,:).*A(:,:,:,6,:));

C(:,:,:,5,:) = ...
    B(:,:,:,1,:).*A(:,:,:,1,:).*A(:,:,:,5,:) + ...
    B(:,:,:,2,:).*A(:,:,:,4,:).*A(:,:,:,6,:) + ...
    B(:,:,:,3,:).*A(:,:,:,3,:).*A(:,:,:,5,:) + ...
    B(:,:,:,4,:).*(A(:,:,:,1,:).*A(:,:,:,6,:) + A(:,:,:,5,:).*A(:,:,:,4,:)) + ...
    B(:,:,:,5,:).*(A(:,:,:,1,:).*A(:,:,:,3,:) + A(:,:,:,5,:).*A(:,:,:,5,:)) + ...
    B(:,:,:,6,:).*(A(:,:,:,3,:).*A(:,:,:,4,:) + A(:,:,:,5,:).*A(:,:,:,6,:));

C(:,:,:,6,:) = ...
    B(:,:,:,1,:).*A(:,:,:,4,:).*A(:,:,:,5,:) + ...
    B(:,:,:,2,:).*A(:,:,:,2,:).*A(:,:,:,6,:) + ...
    B(:,:,:,3,:).*A(:,:,:,3,:).*A(:,:,:,6,:) + ...
    B(:,:,:,4,:).*(A(:,:,:,2,:).*A(:,:,:,5,:) + A(:,:,:,6,:).*A(:,:,:,4,:)) + ...
    B(:,:,:,5,:).*(A(:,:,:,3,:).*A(:,:,:,4,:) + A(:,:,:,6,:).*A(:,:,:,5,:)) + ...
    B(:,:,:,6,:).*(A(:,:,:,2,:).*A(:,:,:,3,:) + A(:,:,:,6,:).*A(:,:,:,6,:));

