function [mrSESSION,ok] = EditSession(mrSESSION,editable)
% Allow user to edit the fields of the session structure
%
%   mrSESSION = EditSession(mrSESSION);
% 
% This function brings up a window that helps the user adjust the mrSESSION
% fields.  
%
% [mrSESSION] relevant to the creation of time-series data --
% "recon". The relevant fields in the session structure are
% a few of the top-level fields and a few of the scan-indexed fields
% within the functionals struct array field. Various other fields
% are displayed for informational purposes.
%
% TODO:  mrSESSION needs to be treated with more respect.  We need a
% sessionGet() and sessionSet() and sessionCreate() functions.  We need to
% touch mrSESSION only through those functions

if ~exist('editable','var'), editable = 1; end

% The following two cell arrays determine the field names to
% display, their descriptive label, and whether or not they are
% to be editable. Each row of the array has the form:
% {field name, label, edit flag}.
topFields = { ...
        'description', 'Description', 1; ...
        'subject', 'Subject', editable ...
    };

inplaneFields = { ...
        'FOV', 'FOV', 0; ...
        'fullSize', 'Anatomy image dimensions', 0; ...
        'voxelSize', 'Anatomy voxel dimensions (mm)', 0; ...
        'nSlices', 'Number of slices', 0 ...
    };
        
scanFields = { ...
        'PfileName', 'Pfile name', 0; ...
        'framePeriod', 'Frame interval (s)', 0; ...
        'totalFrames', 'Total frames in file', 0; ...
        'junkFirstFrames', 'Initial frames to discard', editable; ...
        'nFrames', 'Subsequent frames to retain', editable; ...
        'slices', 'Slices (subset of inplanes)', editable; ...
        'fullSize', 'Functional image dimensions', 0; ...
        'voxelSize', 'Functional voxel dimensions (mm)', 0; ...
        'effectiveResolution', 'Effective resolution (mm)', 0 ...
    };

% Create the top-level fields
if isunix,   fontSize = 10;
else         fontSize = 9;
end

titleFontSize = 12;
butWidth=10;
marginFields = 8.5;
topMargin = 1.5;
versionLength = 25;

%scale factors for x and y axis coordinates
xs = 1.45;  
ys = 1.6;
height = 1 * ys;
vSkip = 0.10 * ys;
dy = (height + vSkip);
x = 1 * xs;

iScan = 1;
topData     = CreateEditData(topFields, mrSESSION);
inplaneData = CreateEditData(inplaneFields, mrSESSION.inplanes);
scanData    = CreateEditData(scanFields, mrSESSION.functionals(iScan));
nTopFields  = length(topData);
nInplaneFields = length(inplaneData);
nScanFields = length(scanData);
numFields   = nTopFields + nInplaneFields + nScanFields; % Add extra line for recon version
maxWidth    = max([topData.width, inplaneData.width, scanData.width]);
pos = [35, 3, maxWidth, (numFields+marginFields)] .* [2, ys, 1.1, dy];

% Create the figure:
topH = figure(...
    'MenuBar', 'none', ...
    'Units', 'char', ...
    'Position', pos, ...
    'Resize','on', ...
    'Name', 'fMRI Reconstruction', ...
    'NumberTitle','off' ...
    );
bkColor = get(topH, 'color');
topHS = num2str(topH);

% center the figure in the screen (ras, 03/06):
set(topH, 'Units', 'normalized');
normPos = get(topH, 'Position'); % fig pos relative to screen
normPos(1:2) = [.5 .5] - normPos(3:4)/2;
set(topH, 'Position', normPos);

% Session parameters:
y = (numFields + marginFields - topMargin) * dy;
titleStr = 'Session Parameters';
uicontrol( ...
    'Style', 'text', ...
    'Units', 'char', ...
    'String', titleStr, ...
    'BackgroundColor', bkColor, ...
    'Position', [0, y, length(titleStr)*xs*1.5, height*1.2], ...
    'HorizontalAlignment', 'left', ...
    'FontWeight', 'bold', ...
    'FontSize', titleFontSize ...
    );

% maxLabelWidth = 0.9 * max([size(char(topData.label), 2), size(char(scanData.label), 2)]) * xs;
maxLabelWidth = 30; % set it to be constant # characters... (ras 03/06)
y = y - dy;

% Create the top-level fields:
for iField=1:nTopFields
  labelPos = [x, y, length(topData(iField).label)*xs, height];
  h = CreateEditRow(topData(iField), labelPos, maxLabelWidth, topH, fontSize, [], 35);
  topData(iField).handle = h;
  y = y - dy;
end

% Inplanes:
y = y - dy;
titleStr = 'Inplanes';
uicontrol( ...
    'Style', 'text', ...
    'Units', 'char', ...
    'String', titleStr, ...
    'BackgroundColor', bkColor, ...
    'Position', [0, y, length(titleStr)*xs*1.5, height*1.2], ...
    'HorizontalAlignment', 'left', ...
    'FontWeight', 'bold', ...
    'FontSize', titleFontSize ...
    );
maxLabelWidth = 0.9 * max([size(char(inplaneData.label), 2), ...
	size(char(inplaneData.label), 2)]) * xs;
y = y - dy;
% Create the top-level fields:
for iField=1:nInplaneFields
  labelPos = [x, y, length(inplaneData(iField).label)*xs, height];
  h = CreateEditRow(inplaneData(iField), labelPos, maxLabelWidth, topH, fontSize);
  inplaneData(iField).handle = h;
  y = y - dy;
end

% Functionals:
y = y - dy;
titleStr = 'Functionals';
uicontrol( ...
    'Style', 'text', ...
    'Units', 'char', ...
    'String', titleStr, ...
    'BackgroundColor', bkColor, ...
    'Position', [0, y, length(titleStr)*xs*1.5, height*1.2], ...
    'HorizontalAlignment', 'left', ...
    'FontWeight', 'bold', ...
    'FontSize', titleFontSize ...
    );
y = y - dy;
% Create the scan-number selection buttons and indicator field:
bpos = [length(titleStr)/2 - 4, y+0.25*dy, 5, height];
cString = ['IncEdit(', topHS, ', -1);'];
uicontrol( ...
    'Style', 'pushbutton', ...
    'String', '<<', ...
    'Units', 'char', ...
    'Position', bpos, ...
    'Callback', cString ...
    );
cString = ['IncEdit(', topHS, ', 0);'];
hScan = uicontrol( ...
    'Style', 'edit', ...
    'Units', 'char', ...
    'String', '1', ...
    'BackgroundColor', [1, 1, 1], ...
    'Position', [length(titleStr)/2, y+0.25*dy, 3*xs+1, height], ...
    'HorizontalAlignment', 'center', ...
    'FontSize', fontSize, ...
    'Callback', cString ...
    );
bpos = [length(titleStr)/2 + 4, y+0.25*dy, 5, height];
cString = ['IncEdit(', topHS, ', 1);'];
uicontrol( ...
    'Style', 'pushbutton', ...
    'String', '>>', ...
    'Units', 'char', ...
    'Position', bpos, ...
    'Callback', cString ...
    );
bpos = [length(titleStr)/2+10, y+0.25*dy, 8, height];
cString = ['DupEdit(', topHS, ');'];
uicontrol( ...
    'Style', 'pushbutton', ...
    'String', 'Copy>>', ...
    'Units', 'char', ...
    'Position', bpos, ...
    'Callback', cString, ...
    'Tooltip','Copy to all subsequent scans' ...
    );
y = y - dy;
% Create the scan-related fields:
for iField=1:nScanFields
  labelPos = [x, y, length(scanData(iField).label)*xs, height];
  h = CreateEditRow(scanData(iField), labelPos, maxLabelWidth, topH, fontSize);
  scanData(iField).handle = h;
  y = y - dy;
end

% Build the UI data structure and attach it to the figure:
uiData.session = mrSESSION;
uiData.original = mrSESSION;
uiData.topData = topData;
uiData.inplaneData = inplaneData;
uiData.scanData = scanData;
uiData.hScan = hScan;
uiData.iScan = iScan;
set(topH, 'UserData', uiData);

% Finally, install the file-control buttons
y = y - dy/2;
butWidth = 11;
bpos = [x, y, butWidth, height*1.2];
uicontrol( ...
    'Style', 'pushbutton', ...
    'String', 'Accept', ...
    'Units', 'char', ...
    'HorizontalAlignment', 'center', ...
    'Position', bpos, ...
    'Callback', 'uiresume', ...
    'FontSize', titleFontSize ...
    );
x = x + butWidth + 1;
bpos = [x, y, butWidth, height*1.2];
uicontrol( ...
    'Style', 'pushbutton', ...
    'String', 'Revert', ...
    'Units', 'char', ...
    'HorizontalAlignment', 'center', ...
    'Position', bpos, ...
    'FontSize', titleFontSize, ...
    'Callback', ['RevertEdit(', topHS, ');'] ...
    );
x = x + butWidth + 1;
bpos = [x, y, butWidth, height*1.2];
uicontrol( ...
    'Style', 'pushbutton', ...
    'String', 'Cancel', ...
    'Units', 'char', ...
    'HorizontalAlignment', 'center', ...
    'Position', bpos, ...
    'FontSize', titleFontSize, ...
    'Callback', ['CancelEdit(', topHS, ');'] ...
    );

% Wait until we get a uiresume, then perform an update. Repeat
% this cycle until the update reports no errors.
ok = 0;
while ~ok
  uiwait(topH);
  uiData = get(topH, 'UserData');
  if isfield(uiData, 'cancel')
    ok=0;
    close(topH);
    return
  end
  ok = UpdateEdit(topH);
end

% After the update is successful, unpack the data
% into the session structure and clean up.
uiData = get(topH, 'UserData');
mrSESSION = uiData.session;
close(topH);
