function searchlight = slInit(varargin)
% slInit(varargin)
% SEARCHLIGHT - INITIALIZE 
% ---------------------------------------------------------
% Initializes a searchlight structure.  Used to run a searchlight using
% slRun.
%
% INPUTS
%   OPTIONS
%       'Path' - Path to directory containing valid mrSESSION file.
%           DEFAULT: Prompt user.
%       'GrowBy' - Measure of searchlight size (circle of x mm radius on
%           surface or Gray View, (x*2+1)^3 voxel cube in Inplane View)
%           DEFAULT: 5
%       'Measure' - What measurements are retrieved for classification.
%           DEFAULT: 'Betas'
%           'Betas'
%           'PeakBsl'
%           'TimeSeries'
%       'SearchView' - DEFAULT: 'Inplane'
%           'Gray'
%           'Inplane' 
%       'RunOptions' - Structure generated by svmRunOptions (if the default
%           parameters are insufficient)
%       'Relabel' - Cell array of condition groupings {[1 3 5]; [2 4 6]}
%       'Remove' - Vector of conditions to remove - occurs post-relabeling.
%           Be wary of using in combination with relabel.
%       'DataType' - DEFAULT: 'MotionComp'
%           'MotionComp'
%           'Original'
%           ... fill in mrVista data type
%       'Processes' - Number of total portions to break searchlight into.
%           DEFAULT: 1
%       'Partitions' - Number of smaller-sized portions to break each
%           portion into.  Allows more incremental saving of progress on
%           each job (e.g. 4 CPUs working on searchlight, each iterating
%           over 30 partitions). DEFAULT: 1
%
% OUTPUTS
%   searchlight - Structure to be passed to slRun.
%
% USAGE
%   Setting up a 5mm Gray view searchlight to be broken up into 30
%   bite-sized portions.
%       searchlight = slInit('Path', '/myexperiment/subject1/', ...
%           'GrowBy', 5, 'SearchView', 'Gray', 'Processes', 30);
%
%   Using relabeling to group odd and even conditions before running the
%   searchlight.
%       searchlight = slInit('Path', '/myexperiment/subject1/', ...
%           'GrowBy', 5, 'SearchView', 'Gray', 'Processes', 30, ...
%           'Relabel', {[1 3 5]; [2 4 6]});
%
% See also SLRUN, SLCOMPLETE, SLSPAWN, SVMINIT.
%
% renobowen@gmail.com [2010]
%

    %%
    searchlight = [];
	path = [];
    growBy = 5;
    measure = 'betas';
    searchView = 'inplane';
    runOptions = [];
    relabel = [];
    remove = [];
    dataType = 'MotionComp';
    processes = 1;
    partitions = 1;
    
    %%
    i = 1;
    while (i <= length(varargin))
        if (isempty(varargin{i})), break; end
        switch (lower(varargin{i}))
            case {'path'}
                path = varargin{i + 1};
            case {'growby'}
                growBy = varargin{i + 1};
            case {'measure'}
                measure = varargin{i + 1};
            case {'searchview'}
                searchView = varargin{i + 1};
            case {'runoptions'} % careful with this one and the next - we assume you know what you're doing with the options struct
                runOptions = varargin{i + 1};
            case {'relabel'}
                relabel = varargin{i + 1};
            case {'remove'}
                remove = varargin{i + 1};
            case {'datatype'}
                dataType = varargin{i + 1};
            case {'processes'}
                processes = varargin{i + 1};
            case {'partitions'}
                partitions = varargin{i + 1};
            otherwise
                fprintf(1, 'Unrecognized option: ''%s''\n', varargin{i});
                return;
        end
        i = i + 2;
    end
    
    if (~isempty(remove) && ~isempty(relabel))
        fprintf(1, 'Warning: Relabeling happens before removing, and given the changes it makes to condition numbers, might lead to undefined behavior.\n');
    end
    
    %%
    % Catch missing session path
	if (isempty(path))
    	[path] = uigetdir(pwd, 'Select Session Path');
        if (~path), return; end
    end
    
    % Set up temp folder for progress files
    tmpDir = sprintf('sl_%d_%s_%s', growBy, measure, searchView);
    if (~isempty(relabel)), tmpDir = [tmpDir '_relabeled']; end
    if (~isempty(remove)), tmpDir = [tmpDir '_reduced']; end
    
    if (~exist(sprintf('%s/%s', path, tmpDir), 'dir'))
        mkdir(sprintf('%s/%s', path, tmpDir));
    end
    
    % Change into path and save out current directory for later restoration
    curDir = pwd;
    cd(path);
    
    % Tell mrVista to shut up, independent of your current prefs
    curVerbose = prefsVerboseCheck;
    setpref('VISTA', 'verbose', 0);

    %%
    if (isempty(runOptions)), runOptions = svmRunOptions(); end
     
    % Initialize views
    vw_ip = initHiddenInplane;
    if (strcmpi(searchView, 'gray')), vw_gray = initHiddenGray; end

    % Vectors/values to generate all inplane coords
    global mrSESSION
    data = mrSESSION.functionals(1);
    sz = data.cropSize;
    
    xlen = sz(1);
    ylen = sz(2);
    zlen = length(data.slices);
    dims  = [xlen; ylen; zlen];
    
    xvals = 1:xlen;
    yvals = 1:ylen;
    zvals = 1:zlen;
    total = xlen * ylen * zlen;

    % Generate 3 x nCoords matrix of all coordinates in inplane space
    xCoords = reshape(repmat(xvals, [ylen*zlen 1]), [1 xlen*ylen*zlen]);
    yCoords = repmat(reshape(repmat(yvals, [zlen 1]), [1 ylen*zlen]), [1 xlen]);
    zCoords = repmat(zvals, [1 xlen*ylen]);
    allCoords = [xCoords; yCoords; zCoords];

    % Create SVM struct with whole measured brain ROI
    rs = upSampleFactor(vw_ip);
	vw_ip.selectedROI = 1;
    vw_ip.ROIs(1).name = 'searchlight';
    vw_ip.ROIs(1).coords = allCoords .* repmat(rs', [1 total]);
    svm = svmInit('view', vw_ip, 'measure', measure, 'datatype', dataType, 'preservecoords', 1);
    if (~isempty(relabel)); svm = svmRelabel(svm, relabel); end
    if (~isempty(remove)); svm = svmRemove(svm, remove); end
    
    % Generate start/end indices for partitions to process in parallel
    % Also store specific info
    if (strcmpi(searchView, 'gray'))
        nTrials = size(svm.data, 1);
        newGrayData = [];
        for i = 1:nTrials
            vw_ip.map = { populateMap(svm.data(i, :), svm.coordsInplane) };
            vw_gray = ip2volParMap(vw_ip, vw_gray, 1, [], [], 1);
            newGrayData = [newGrayData; vw_gray.map{1}];
        end
        
        svm.data        = newGrayData;
        svm.coordsGray  = vw_gray.coords;
        svm = rmfield(svm, 'coordsAnatomy');
        svm = rmfield(svm, 'coordsInplane');

        ipInds = [  vw_gray.inplaneLeftIndices', vw_gray.inplaneRightIndices'; ...
                    zeros(size(vw_gray.inplaneLeftIndices')), ...
                    ones(size(vw_gray.inplaneRightIndices'))];
                
        nElems = size(ipInds, 2);
    else
        nElems = total;
    end
    
    nPerFold        = floor(nElems / processes);
    tmp             = ones(1, processes) * nPerFold;
    startInds       = tmp .* (0:(processes - 1)) + 1;
    endInds         = tmp .* (1:processes);
    endInds(end)    = nElems;

    %%
    % Save out all other necessary parameters
    searchlight.path            = path;
    searchlight.tmpDir          = tmpDir;
    searchlight.curDir          = curDir;
    searchlight.growBy          = growBy;
    searchlight.searchView      = searchView;
    searchlight.runOptions      = runOptions;
    searchlight.svm             = svm;
    searchlight.dims            = dims;
    searchlight.allCoords       = allCoords;    
    searchlight.startInds       = startInds;
    searchlight.endInds         = endInds;
    searchlight.processes       = processes;
    searchlight.partitions      = partitions;
    searchlight.vw_ip           = vw_ip;

    if (strcmpi(searchView, 'gray'))
        searchlight.vw_gray     = vw_gray;
        searchlight.ipInds      = ipInds;
    end
    
    setpref('VISTA', 'verbose', curVerbose);

end

function map = populateMap(values, coords)
	sz = [max(coords(1,:)) max(coords(2,:)) max(coords(3,:))];
    map = zeros(sz);
	for i = 1:length(coords)
		map(coords(1, i), coords(2, i), coords(3, i)) = values(i);
	end
end


