function vw = makeIsoEccenROIs(vw)
%
% vw = makeIsoEccenROIs(vw)
%
% Creates iso-eccentricity ROIs from an Atlas. The ROIs are created based
% on the ring map.
% The user will be prompted for params if the atlasImage is not passed in.
% 
% HISTORY:
%   2002.04.01 RFD (bob@white.stanford.edu): wrote it.
%  2002.10.22 RFD cleaned up code, fixed ROI field bug, and a matlab 6.5
%  glitch (we now have to force our logical masks to be double arrays).

% the convention is to make the wedge atlas scan number 1 and the ring
% atlas scan number 2
ringAtlasScanNum = 2;
global dataTYPES;

hemisphere =  viewGet(vw, 'Current Slice');

if(hemisphere==1), hemiString = 'left';
else               hemiString = 'right';
end

prompt = {'num ROIs:', 'ROI value:', 'eccenRadRangePerStep:'};
default = {'8','0','0.05'};
answer = inputdlg(prompt, 'Iso-eccentricity Parameters', 1, default, 'on');

if ~isempty(answer)
    viewA = getAtlasView;
    if(isempty(viewA))
        myErrorDlg('Sorry- there are no Atlas FLAT windows open.');
    end
    numOfEccens = str2num(answer{1});
    ROISelection = str2num(answer{2});
    eccenRadRangePerStep = str2num(answer{3});
    atlasImage = squeeze(viewA.ph{ringAtlasScanNum}(:,:,hemisphere));
    ROIsImage = squeeze(viewA.co{ringAtlasScanNum}(:,:,hemisphere));
else
    myErrorDlg('You need to provide some valid parameters!');
end

%%Turned off 'Full Match' for Atlas names - AAB
atlasTypeNum = existDataType('Atlases',[],0);
if(atlasTypeNum == 0)
    myErrDlg('No Atlases data type!');
end
phaseShift = dataTYPES(atlasTypeNum).atlasParams(ringAtlasScanNum).phaseShift(hemisphere);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%  calculate ROI coordinates for original image
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

imsize = size(ROIsImage);
colorList = {'yellow','magenta','cyan','red','green','blue'};
colorChar = char(colorList)';
colorChar = colorChar(1,:);

% force logical mask to be double array, so it can take NaNs. (matlab 6.5
% requires this)
ROIsTemplate = double(ROIsImage == ROISelection);
ROIsTemplate(ROIsTemplate>0) = 1;
ROIsTemplate(ROIsTemplate==0) = NaN;

% get coordinates for angle (intervals) from the warped atlas image
eccenRadMin = min(min(ROIsImage));
eccenRadMax = max(max(ROIsImage));

if exist('eccenRadRangePerStep', 'var')
    ROINo = 0;
    % We set the eccen spacing to approximate cortical magnification.
    eccenRangeLo = logspace(log10(.1), log10(2*pi-eccenRadRangePerStep), numOfEccens);
    eccenRangeLo = mod(eccenRangeLo + phaseShift, 2*pi);
    eccenRangeHi = logspace(log10(.1+eccenRadRangePerStep), log10(2*pi), numOfEccens);
    eccenRangeHi = mod(eccenRangeHi + phaseShift, 2*pi);
    for ii=1:length(eccenRangeLo)
        % find coordinates 
        [yy,xx] = find( (atlasImage.*ROIsTemplate) <= eccenRangeHi(ii) & ...
            atlasImage >= eccenRangeLo(ii));
        numOfCoords = length(xx);
        if(numOfCoords>0)
            ROINo = ROINo + 1;
            xCoords=reshape(xx,1,numOfCoords);
            yCoords=reshape(yy,1,numOfCoords);

            % circle through the given colors
            ROIs(ROINo).color = colorChar(mod(ROINo-1,length(colorChar))+1);
            % store coordinates in ROIs structure
            ROIs(ROINo).coords = [ yCoords ; xCoords ; hemisphere*ones(size(xCoords))];
            ROIs(ROINo).name = [hemiString,'_eccen', num2str(ROINo)];
            ROIs(ROINo).viewType = 'FLAT';
        end
    end
end

for(ii=1:length(ROIs))
    vw = addROI(vw, ROIs(ii));
end

refreshView(vw);

return;

% debugging: show warped ROI image
%     im=zeros(size(ROIImage));
%     numOfCoords = length(yy);
%     for i=1:1:numOfCoords
%         y = yy(i);  
%         x = xx(i); 
%         im(y,x) = 1;
%     end
%debugging end: show warped ROI image