function rxXformTSeries(rx,sessionDir,scans,base,srcDataType,tgtDataType);
%
% rxXformTSeries(rx,sessionDir,scans,[base],[srcDataType],[tgtDataType]);
%
% Apply a transformation generated by mrRx to tSeries
% from a mrVista data type. Store the results in a new
% data type.
%
% Hopefully this will serve as an effective fallback for
% motion correction, when automatic algorithms fail to fix
% all the between-scans motion.
%
% rx: a mrRx prescription struct.
%
% sessionDir: directory containing the session's
% mrSESSION.mat file.
%
% scans: scans from the source data type to which to apply
% the transform.
%
% base: if passed as a non-empty value, the number of the
% base scan to which the transformed scans have been compared.
% Will copy this scan, unchanged, from the source to the 
% target data type before xforming the others.
%
% srcDataType: data type from which to take the series. 
%              [Default: prompt user]
%
% tgtDataType: data type under which to save the new tSeries.
%              [default 'ManualMC']
%
%
%
% 03/05 ras.
if ieNotDefined('rx')
    cfig = findobj('Tag','rxControlFig');
    rx = get(cfig,'UserData');
end

mrGlobals;
loadSession;
names = {dataTYPES.name};

if ieNotDefined('srcDataType')
    % have user select points
    [whichDt, ok] = listdlg('PromptString','Apply Xform to Which Data Type?',...
                                'ListSize',[400 600],...
                                'ListString',names,...
                                'InitialValue',1 ,...
                                'SelectionMode','single',...
                                'OKString','OK');
                            
    % exit gracefully if canceled                
	if ~ok  return;  end
        
    srcDataType = dtNames{whichDt};
end

if ieNotDefined('base')
    base = [];
end

if ieNotDefined('tgtDataType')
	def = {'ManualMC'};
	resp = inputdlg({'Name for new data type?'}, mfilename, 1, def);
	tgtDataType = resp{1};
end

% check if the source, target data types exist
if ischar(srcDataType)
	srcNum = cellfind(names,srcDataType);
	if isempty(srcNum)
        msg = sprintf('Source data type %s not found.',srcDataType);
        error(msg);
	end
elseif isnumeric(srcDataType)
    srcNum = srcDataType;
end

if ieNotDefined('scans')
    scanNames = {dataTYPES(dtNum).scanParams.annotation};
    
    % have user select points
    [scans, ok] = listdlg('PromptString','Apply Xform to Which Scans?',...
                                'ListSize',[400 600],...
                                'ListString',scanNames,...
                                'InitialValue',1 ,...
                                'SelectionMode','single',...
                                'OKString','OK');
                            
    % exit gracefully if canceled                
	if ~ok  return;  end
    
elseif scans==0
	% flag for all scans in data type
	scans = 1:length(dataTYPES(srcNum).scanParams);
	
end

callingDir = pwd;

cd(sessionDir);

if ~exist('mrSESSION.mat','file')
    error('No mrSESSION.mat file in specified directory.')
end

% init session; check mrVista not already running
mrGlobals
% if ~isempty(dataTYPES)
%     msg = 'You seem to be running a mrVista session already. ';
%     msg = [msg 'Try running this when you''re done, or else '];
%     msg = [msg 'use mrvCleanWorkspace to clear up the variables.'];
%     myErrorDlg(msg);
% end
loadSession;
HOMEDIR = pwd;
hI = initHiddenInplane;


tgtNum = cellfind(names,tgtDataType);

if isempty(tgtNum) 
    % initialize new data type
    tgtNum = length(dataTYPES)+1;
    dataTYPES(tgtNum).name = tgtDataType;    
    tgtScans = 1:length(scans);
else
    N = length(dataTYPES(tgtNum).scanParams);
    tgtScans = N+1:N+length(scans);
end
    
% get ready to go
tic

% copy any selected base data types
if ~isempty(base)
    % get scan # of base in new dt
    newBaseNum = tgtScans(1);
    
    % adjust # of xformed target scans
    tgtScans = tgtScans + 1;
    
    % also need to copy base data types
	dataTYPES(tgtNum).scanParams(newBaseNum) = ...
        dataTYPES(srcNum).scanParams(base);
	dataTYPES(tgtNum).blockedAnalysisParams(newBaseNum) = ...
        dataTYPES(srcNum).blockedAnalysisParams(base);
	dataTYPES(tgtNum).eventAnalysisParams(newBaseNum) = ...
        dataTYPES(srcNum).eventAnalysisParams(base);
    
    hwait = waitbar(0,'Copying base tSeries...');
    for slice = 1:numSlices(hI)
        hI = selectDataType(hI,srcDataType);
        tSeries = loadtSeries(hI,base,slice);        
        hI = selectDataType(hI,tgtDataType);
        savetSeries(tSeries,hI,newBaseNum,slice);
        waitbar(slice/numSlices(hI),hwait);
    end
    close(hwait);
    clear tSeries
end


% copy scan params to new data type
dataTYPES(tgtNum).scanParams(tgtScans) = ...
    dataTYPES(srcNum).scanParams(scans);
dataTYPES(tgtNum).blockedAnalysisParams(tgtScans) = ...
    dataTYPES(srcNum).blockedAnalysisParams(scans);
dataTYPES(tgtNum).eventAnalysisParams(tgtScans) = ...
    dataTYPES(srcNum).eventAnalysisParams(scans);

% main loop, across scans
for s = 1:length(scans)
    scan = scans(s);
	hI = selectDataType(hI,srcDataType);
    tMat = tSeries4D(hI,scan);

	xsz = size(tMat,1);
	ysz = size(tMat,2);
	zsz = size(tMat,3);
    
    % build coords for xform
    xrng = (1:xsz) - round(xsz/2);
    yrng = (1:ysz) - round(ysz/2);
    zrng = (1:zsz) - round(zsz/2);
    [X Y Z] = meshgrid(xrng,yrng,zrng);
    coords = [X(:) Y(:) Z(:) ones(size(Z(:)))]';

	% get new coords for the xformed frames
	newCoords = rx.xform * coords;
    
    hwait = waitbar(0,sprintf('Xforming Scan %i/%i',s,length(scans)));
    
	% interpolate
    for t = 1:size(tMat,4)		
%         subvol = tMat(:,:,:,t);
% 		interpVals = myCinterp3(subvol,[xsz ysz],zsz,newCoords(1:3,:)',0);
% 		tMat(:,:,:,t) = reshape(interpVals,[xsz ysz zsz]);
		
		tMat(:,:,:,t) = warpAffine3(tMat(:,:,:,t), rx.xform, NaN, 1);		
        
        waitbar(t/size(tMat,4),hwait);
    end
    
    % save tSeries
    waitbar(1,hwait,'Saving tSeries...');
	hI = selectDataType(hI,tgtDataType);
    save4DTSeries(hI,tMat,tgtScans(s));
    
    close(hwait);
end

saveSession;

fprintf('Done. Time elapsed: %4.0f min %2.2f sec.\n',toc/60,mod(toc,60));


cd(callingDir);

return
