function ampMap = MapLaminarProfiles(scan, deltaThick)

% ampMap = MapLaminarProfiles([scan, deltaThick]);
%
% Calculate the laminar amplitude profile of the specified laminarIndex
% map. The calculation is performed only in the VOLUME view. Returns the
% mean value, std. err., and bin centers as a 3 x N array in each non-empty
% cell corresponding to the original laminarIndex map. The calculation is
% performed for the designated input scan; default value is the current
% scan. The profile is calculated on depth increments specified by the
% optional deltaThick input parameter; default value is the volume voxel
% size.  OUTPUT: a cell vector with a length equal to the number of layer-1
% voxels in the the current Gray view.
%
% Ress, 11/04
% Ress, 10/05 Cleaned up and improved documentation. 

mrGlobals

LoadLaminarIndices;

% Get selected volume view, put it in 'Gray' mode if necessary:
selectedVOLUME = viewSelected('volume'); 
view = VOLUME{selectedVOLUME};
if strcmp(view.viewType, 'Gray'), view = switch2Vol(view); end

ampMap = cell(size(view.laminarIndices));
if isempty(ampMap)
  Alert('No laminar coordinates!');
  return
end

if ~exist('scan', 'var'), scan = getCurScan(view); end
if ~exist('deltaThick', 'var'), deltaThick = mean(view.mmPerVox); end

if ~isfield(view, 'laminae')
  view = switch2Vol(view);
  view = loadLaminae(view);
  if ~isfield(view, 'laminae'), return, end
end

% Get amplitudes.
if isempty(view.amp), view = loadCorAnal(view); end

if ~isempty(view.amp)
  amp = view.amp{scan};
  ph = view.ph{scan};
else
  Alert('No amplitudes!')
  return
end

% Put the global view into the new state:
VOLUME{selectedVOLUME} = view;

% Initialize utility volume and its coordinates:
dims = size(view.anat);
vol = repmat(NaN, dims);
vInds = coords2indices(view.coords, dims);

% Main calculation loop:
nCells = length(view.laminarIndices(:));
waitH = waitbar(0, 'Calculating laminar amplitude profile map...');
for ii=1:nCells
  waitbar(ii/nCells, waitH);
  
  % Each element of laminarIndex corresponds to a layer-1 voxel, and
  % contains volume indices denoting a set of volume indices . These
  % indices, in turn, correspond to an extension of that layer-1 voxel
  % inward and outward along the normal directions associated with its
  % nearest gray-white interface vertex that was generated by mrMesh. See
  % MapLaminarIndices.m for more details. 
  inds = view.laminarIndices{ii};
  if ~isempty(inds)
    % Associate the indices with laminar distance map values:
    vol(vInds) = view.laminae;
    tVals = vol(inds)  * mean(view.mmPerVox);
    % Associate the indices with (possibly complex) amplitude values
    vol(:) = NaN;
    vol(vInds) = amp;
    ampVals = vol(inds);
    if ~isempty(ph)
      vol(vInds) = ph;
      phVals = vol(inds);
    end
    % Remove indices that extend beyond the data region.
    nEmpty = sum(isnan(ampVals));
    if nEmpty > 0
      ok = isfinite(ampVals);
      ampVals = ampVals(ok);
      if ~isempty(ph), phVals = phVals(ok); end
      tVals = tVals(ok);
    end
    % Deal with potentially complex amplitude data:
    if ~isempty(ph)
      zVals = ampVals .* exp(i*phVals);
    else
      zVals = ampVals;
    end
    % Set up the laminar binning loop
    tRange = max(tVals) - min(tVals);
    nHist = ceil(tRange / deltaThick);
    mAmps = zeros(nHist, 1);
    eAmps = zeros(nHist, 1);
    t = zeros(nHist, 1);
    minT = min(tVals);
    for iH=1:nHist
      % For each laminar bin, collect all amplitude values. The absolute
      % value of their mean is then used as the data point for that bin
      % center, while the absolute value of the complex standard error is
      % used as an error estimate. The bin center laminar coordinate is
      % also recorded.
      maxT = minT + deltaThick;
      binInds = find((tVals >= minT) & (tVals < maxT));
      if isempty(binInds)
        mAmps(iH) = NaN;
        eAmps(iH) = 0;
      else
        z = zVals(binInds);
        mAmps(iH) = abs(mean(z));
        eAmps(iH) = std(z)/sqrt(length(z));
      end
      t(iH) = 0.5 * (minT + maxT);
      minT = maxT;
    end
    ampMap{ii} = [mAmps, eAmps, t];
  end
end
close(waitH)